using UnityEngine;
using System.Collections;
using System.Collections.Generic;

public class ServerScript : MonoBehaviour
{
	enum STATE{NEUTRAL,DUCK,JUMP,DUCK_PUNCH,JUMP_PUNCH,PUNCH};
	enum ACTION{NONE,DUCK,JUMP,PUNCH};
	enum DAMAGE{NONE,LOW,MID,HIGH};
	class PlayerState {
		public int health;
		public STATE state;
		public ACTION nextAction;
		public int idx;
		public bool hitMissed;
		public NetworkPlayer net;
		
		public void reset(){
			health=50;
			state=STATE.NEUTRAL;
			nextAction = ACTION.NONE;
			idx=0;
			hitMissed = false;
		}
	}
	
	List<NetworkPlayer> players = new List<NetworkPlayer>();
	bool playing;
	float maxGameTime = 240f;
	float moveTime = 1f;
	float startTime;
	float lastMoveTime;
	int disconnectGame = 0;
	PlayerState p1 = new PlayerState();
	PlayerState p2 = new PlayerState();
	ACTION p1Action;
	ACTION p2Action;

	// Use this for initialization
	void Start ()
	{
		Network.InitializeServer(20, 21125, false);
	}
	
	// Update is called once per frame
	void Update ()
	{
		if(playing){
			//Check if the game is over
			if(Time.time - startTime > maxGameTime || p1.health <=0 || p2.health <= 0 || disconnectGame!=0){
				Debug.Log("Game Over");
				if((p1.health<=0 && p2.health<=0) || p1.health==p2.health)
					Debug.Log("Draw");
				if(p1.health < p2.health)
					Debug.Log("P2 wins!");
				if(p2.health < p1.health)
					Debug.Log("P1 wins!");
				playing = false;
				if(disconnectGame!=1)
					Network.CloseConnection(p1.net,true);
				if(disconnectGame!=2)
					Network.CloseConnection(p2.net,true);
			}
			
			if(Time.time - lastMoveTime > moveTime){
				//Run state machine
				p1Action = p1.nextAction;
				p1.nextAction=ACTION.NONE;
				p2Action = p2.nextAction;
				p2.nextAction=ACTION.NONE;
				simulateFrame();
				networkView.RPC("updateState",p1.net,(int)p1Action,p1.health,(int)p1.state,p1.idx,(int)p2Action,p2.health,(int)p2.state,p2.idx);
				networkView.RPC("updateState",p2.net,(int)p2Action,p2.health,(int)p2.state,p2.idx,(int)p1Action,p1.health,(int)p1.state,p1.idx);
				lastMoveTime = Time.time;
				Debug.Log(".");
				Debug.Log(".");
				Debug.Log(".");
				Debug.Log("Time:"+(Time.time-startTime));
				Debug.Log("P1{Action:"+p1Action+" State:"+p1.state+" Index:"+p1.idx+" Health:"+p1.health);
				Debug.Log("P2{Action:"+p2Action+" State:"+p2.state+" Index:"+p2.idx+" Health:"+p2.health);
			}
		}
		else if(players.Count>=2){
			p1.net = players[0];
			p2.net = players[1];
			p1.reset();
			p2.reset();
			players.RemoveRange(0,2);
			
			for(int i=0; i<players.Count; i++)
				networkView.RPC("placeInLine",players[i],players.Count);
			
			playing = true;
			lastMoveTime = Time.time;
			startTime = Time.time;
			disconnectGame = 0;
			Debug.Log("Starting Game!");
		}
		//else
			//Debug.Log("Waiting for players...");
	}
	
	void OnPlayerConnected(NetworkPlayer player){
		players.Add(player);
		networkView.RPC("placeInLine",player,players.Count);
	}
	
	[RPC]
	void placeInLine(int i){
		Debug.Log("The server shouldn't get this message");	
	}
	
	void OnPlayerDisconnected(NetworkPlayer player){
		//check the queue for disconnections
		for(int i=0; i<players.Count; i++){
			if(players[i]==player){
				Debug.Log("A player in the waiting queue disconnected...");
				players.RemoveAt(i);
				return;
			}
		}
		
		if(playing){
			//if we are playing a game check for a player leaving
			if(p1.net == player){
				Debug.Log("P1 disconnected....ending game");
				disconnectGame = 1;
			}
			else if(p2.net == player){
				Debug.Log("P2 disconnected....ending game");
				disconnectGame = 2;
			}
		}
	}
	
	[RPC]
	void updateState(int myAction, int myHealth, int myState, int myIndex, 
	                 int theirAction, int theirHealth, int theirState, int theirIndex){
		Debug.Log("The server shouldn't receive state...");	
	}
	
	[RPC]
	void keyPress(int a, NetworkMessageInfo n){
		if(!playing){
			Debug.Log("Someone is sending actions when a game isn't being played yet...");
			Network.CloseConnection(n.sender,true);
		}
		if(n.sender==p1.net){
			if(a<0 || a>3)
				p1.nextAction = ACTION.NONE;
			else{
				p1.nextAction = (ACTION)a;
				Debug.Log("Got an action from P1:"+p1.nextAction);
			}
		}
		else if(n.sender==p2.net){
			if(a<0 || a>3)
				p2.nextAction = ACTION.NONE;
			else{
				p2.nextAction = (ACTION)a;
				Debug.Log("Got an action from P2:"+p2.nextAction);
			}
		}
		else{
			Debug.Log("Someone is sending actions who isn't playing yet...");
			Network.CloseConnection(n.sender,true);
		}
	}
	
	[RPC]
	void chat(string s, NetworkMessageInfo n){
		if(!playing){
			Debug.Log("Someone is sending chat when a game isn't being played yet...");
			Network.CloseConnection(n.sender,true);
		}
		if(n.sender==p1.net)
			Debug.Log("Got chat from P1:"+s);
		else if(n.sender==p2.net)
			Debug.Log("Got chat from P2:"+s);
		else{
			Debug.Log("Someone is sending chat who isn't playing yet...");
			Network.CloseConnection(n.sender,true);
		}
	}
	
	void simulateFrame(){
		updateState(p1,p1Action);
		updateState(p2,p2Action);
		DAMAGE d1 = getDamage(p1);
		DAMAGE d2 = getDamage(p2);
		dealDamage(d1,p2,p1);
		dealDamage(d2,p1,p2);
	}
	
	//updates the state, and returns the damage that may apply to the opponent (it also assumes hits miss)
	void updateState(PlayerState p, ACTION a){
		switch(p.state){
		case STATE.NEUTRAL:
			runNeutral(p,a);
			return;
		case STATE.PUNCH:
			runPunch(p,a);
			return;
		case STATE.JUMP:
			runJump(p,a);
			return;
		case STATE.JUMP_PUNCH:
			runJumpPunch(p,a);
			return;
		case STATE.DUCK:
			runDuck(p,a);
			return;
		case STATE.DUCK_PUNCH:
			runDuckPunch(p,a);
			return;
		default:
			Debug.Log("Unsupported State: " + p.state);
			return;
		}
	}
	
	DAMAGE getDamage(PlayerState p){
		if(p.state==STATE.PUNCH && p.idx==0)
			return DAMAGE.MID;
		else if(p.state==STATE.JUMP_PUNCH && p.idx==0)
			return DAMAGE.HIGH;
		else if(p.state==STATE.DUCK_PUNCH && p.idx==0)
			return DAMAGE.LOW;
		else
			return DAMAGE.NONE;
	}
	
	void dealDamage(DAMAGE d, PlayerState p, PlayerState dealer){
		if(d==DAMAGE.HIGH &&
		   (p.state==STATE.NEUTRAL ||
		    p.state==STATE.PUNCH ||
		    p.state==STATE.JUMP ||
		    p.state==STATE.JUMP_PUNCH ||
		    (p.state==STATE.DUCK && p.idx!=0) ||
		    (p.state==STATE.DUCK_PUNCH && p.idx!=0))){
			p.health--;
			dealer.hitMissed = false;
		}
		else if(d==DAMAGE.HIGH){
			dealer.hitMissed = true;
			Debug.Log("Jump punch MISSED!!!");	
		}
		else if(d==DAMAGE.MID && 
		        (p.state==STATE.NEUTRAL ||
		         p.state==STATE.PUNCH ||
		         (p.state==STATE.JUMP && p.idx!=0) ||
		         (p.state==STATE.JUMP_PUNCH && p.idx!=0) ||
		         (p.state==STATE.DUCK && p.idx!=0) ||
		         (p.state==STATE.DUCK_PUNCH && p.idx!=0))){
			p.health--;
			dealer.hitMissed = false;
		}
		else if(d==DAMAGE.MID){
			dealer.hitMissed = true;
			Debug.Log("Mid punch MISSED!!!");	
		}
		else if(d==DAMAGE.LOW &&
		        (p.state==STATE.NEUTRAL ||
		         p.state==STATE.PUNCH ||
		         (p.state==STATE.JUMP && p.idx!=0) ||
		         (p.state==STATE.JUMP_PUNCH && p.idx!=0) ||
		         p.state==STATE.DUCK ||
		         p.state==STATE.DUCK_PUNCH)){
			p.health--;
			dealer.hitMissed = false;
		}
		else if(d==DAMAGE.LOW){
			dealer.hitMissed = true;
			Debug.Log("Duck punch MISSED!!!");	
		}
	}
		
	STATE getStateFromAction(ACTION a){
		if(a==ACTION.NONE)
			return STATE.NEUTRAL;
		else if(a==ACTION.PUNCH)
			return STATE.PUNCH;
		else if(a==ACTION.JUMP)
			return STATE.JUMP;
		else if(a==ACTION.DUCK)
			return STATE.DUCK;
		else{
			Debug.Log("Can't get state: Unknown Action");
			return STATE.NEUTRAL;
		}
	}
	
	void runNeutral(PlayerState p, ACTION a){
		p.state = getStateFromAction(a);
		p.idx = 0;
	}
	
	void runPunch(PlayerState p, ACTION a){
		if(p.idx==0){
			if(p.hitMissed)
				p.idx++;
			else{
				p.state = getStateFromAction(a);
				p.idx = 0;
			}
		}
		else if(p.idx==1)
			p.idx++;
		else{ //frame 2
			p.state = getStateFromAction(a);
			p.idx = 0;
		}
	}
	
	void runJump(PlayerState p, ACTION a){
		if(p.idx==0){
			if(a==ACTION.PUNCH){
				p.state = STATE.JUMP_PUNCH;
				p.idx = 0;
			}
			else
				p.idx++;
		}
		else{ //frame 1
			p.state = getStateFromAction(a);
			p.idx = 0;
		}
	}
	
	void runJumpPunch(PlayerState p, ACTION a){
		if(p.idx==0)
			p.idx++;
		else if(p.idx==1){
			if(p.hitMissed)
				p.idx++;
			else{
				p.state = getStateFromAction(a);
				p.idx=0;
			}
		}
		else{ //frame 2
			p.state = getStateFromAction(a);
			p.idx=0;
		}
	}
	
	void runDuck(PlayerState p, ACTION a){
		if(p.idx==0){
			if(a==ACTION.PUNCH){
				p.state = STATE.DUCK_PUNCH;
				p.idx = 0;
			}
			else
				p.idx++;
		}
		else{ //frame 1
			p.state = getStateFromAction(a);
			p.idx = 0;
		}
	}
	
	void runDuckPunch(PlayerState p, ACTION a){
		if(p.idx==0)
			p.idx++;
		else if(p.idx==1){
			if(p.hitMissed)
				p.idx++;
			else{
				p.state = getStateFromAction(a);
				p.idx=0;
			}
		}
		else{ //frame 2
			p.state = getStateFromAction(a);
			p.idx=0;
		}
	}
}
